#version 400 compatibility

/*
====================================================================================================

    Copyright (C) 2020 RRe36

    All Rights Reserved unless otherwise explicitly stated.


    By downloading this you have agreed to the license and terms of use.
    These can be found inside the included license-file
    or here: https://rre36.com/copyright-license

    Violating these terms may be penalized with actions according to the Digital Millennium
    Copyright Act (DMCA), the Information Society Directive and/or similar laws
    depending on your country.

====================================================================================================
*/

/*DRAWBUFFERS:0356*/
layout(location = 0) out vec4 scene;
layout(location = 1) out vec4 lightmapData;
layout(location = 2) out vec3 skyboxData;
layout(location = 3) out vec4 filterAux;

#include "/lib/head.glsl"
#include "/lib/util/encoders.glsl"
#include "/lib/util/colorspace.glsl"

const int noiseTextureResolution = 256;

in vec2 coord;

flat in vec3 cloudLightColor;

flat in mat4x3 lightColor;
flat in mat2x3 skyColorMat;

uniform sampler2D colortex0;
uniform sampler2D colortex1;
uniform sampler2D colortex7;
uniform sampler2D depthtex1;

uniform sampler2D noisetex;

uniform int frameCounter;
uniform int worldTime;

uniform float aspectRatio;
uniform float far, near;
uniform float frameTimeCounter;
uniform float wetness;
uniform float cloudLightFlip;
uniform float worldAnimTime;

uniform vec2 viewSize, pixelSize;
uniform vec2 taaOffset;
uniform vec2 skyCaptureResolution;

uniform vec3 upvec, upvecView;
uniform vec3 sunvec, sunvecView;
uniform vec3 moonvec, moonvecView;
uniform vec3 cameraPosition;
uniform vec3 cloudLightDir, cloudLightDirView;

uniform vec4 daytime;

uniform mat4 gbufferModelView, gbufferModelViewInverse;
uniform mat4 gbufferProjection, gbufferProjectionInverse;


/* ------ includes ------ */
#define FUTIL_LINDEPTH
#define FUTIL_D3X3
#define FUTIL_ROT2
#include "/lib/fUtil.glsl"

#include "/lib/util/transforms.glsl"
#include "/lib/atmos/phase.glsl"
#include "/lib/atmos/skyGradient.glsl"

vec3 getSun(vec3 viewvec, vec3 albedo) {
    vec3 v      = -viewvec;
    vec3 sv     = normalize(sunvecView+v);
    float sun   = dot(sv, v);

    float s   = 1.0-linStep(sun, 0.03, 0.08);

    return albedo*s*lightColor[0]*pi;
}

vec3 getMoon(vec3 viewvec, vec3 albedo) {
    vec3 v      = -viewvec;
    vec3 sv     = normalize(moonvecView+v);
    float sun   = dot(sv, v);

    float s   = 1.0-linStep(sun, 0.03, 0.08);

    return albedo*s*lightColor[2]*4.0;
}

#include "/lib/frag/noise.glsl"

vec3 skyStars(vec3 worldDir) {
    vec3 plane  = worldDir/(worldDir.y+length(worldDir.xz)*0.66);
    float rot   = worldTime*rcp(2400.0);
    plane.x    += rot*0.6;
    plane.yz    = rotatePos(plane.yz, (25.0/180.0)*pi);
    vec2 uv1    = floor((plane.xz)*768)/768;
    vec2 uv2    = (plane.xz)*0.04;

    vec3 starcol = vec3(0.3, 0.78, 1.0);
        starcol  = mix(starcol, vec3(1.0, 0.7, 0.6), noise2D(uv2).x);
        starcol  = normalize(starcol)*(noise2D(uv2*1.5).x+1.0);

    float star  = 1.0;
        star   *= noise2D(uv1).x;
        star   *= noise2D(uv1+0.1).x;
        star   *= noise2D(uv1+0.26).x;

    star        = max(star-0.25, 0.0);
    star        = saturate(star*4.0);

    return star*starcol*0.25*sqrt(daytime.w);
}

/* ------ sky reflection ------ */
vec2 sincos(float x) {
    return vec2(sin(x), cos(x));
}

vec3 projectSphere(vec2 coord) {
    coord  *= vec2(tau, pi);
    vec2 lon = sincos(coord.x) * sin(coord.y);
    return vec3(lon.x, cos(coord.y), lon.y);
}

#include "/lib/frag/gradnoise.glsl"

#include "/lib/atmos/clouds.glsl"

vec4 cloudSystem(vec3 worldDir, float vDotL, float dither, vec3 skyColor) {
    vec3 totalScattering    = vec3(0.0);
    float totalTransmittance = 1.0;

    vec3 sunlight       = (worldTime>23000 || worldTime<12900) ? cloudLightColor : lightColor[2];
        sunlight       *= cloudLightFlip;
    vec3 skylight       = mix(skyColorMat[0], lightColor[1] * (1.0 - sqrt(daytime.w)*0.96), 0.4);

    float pFade         = saturate(mieHG(vDotL, 0.65));

    const float eyeAltitude = 64.0;
    vec3 camPos     = vec3(cameraPosition.x, eyeAltitude, cameraPosition.z);

    bool visibleVol = worldDir.y > 0.0;

    float lightNoise = ditherGradNoiseTemporal();

    #ifdef cloudVolumeStoryMode
        const float sigmaA  = 0.1;
    #else
        const float sigmaA  = 0.2;
    #endif

    const float sigmaT  = 1.25;

    if (visibleVol) {
        vec3 bottom     = worldDir * ((cloudVolume0Alt - eyeAltitude) * rcp(worldDir.y));
        vec3 top        = worldDir * ((cloudVol0MaxY - eyeAltitude) * rcp(worldDir.y));

        if (worldDir.y < 0.0) {
            bottom      = vec3(0.0);
            top         = vec3(0.0);
        }

        vec3 start      = bottom;
        vec3 end        = top;

        float stepCoeff     = 1.0;
        uint steps          = uint(cloudVolume0Samples * stepCoeff);

        vec3 rStep          = (end - start) * rcp(float(steps));
        vec3 rPos           = rStep * dither + start + camPos;
        float rLength       = length(rStep);

        vec3 scattering     = vec3(0.0);
        float transmittance = 1.0;

        for (uint i = 0; i < steps; ++i, rPos += rStep) {
            if (transmittance < 0.01) break;
            if (rPos.y < cloudVolume0Alt || rPos.y > cloudVol0MaxY) continue;

            float dist  = distance(rPos, camPos);
            if (dist > cloudVolume0Clip) continue;

            float density = cloudVolume0Shape(rPos);
            if (density <= 0.0) continue;

            float extinction    = density * sigmaT;
            float stepT         = exp(-extinction * rLength);
            float integral      = (1.0 - stepT) * rcp(sigmaT);

            vec3 stepScatter    = vec3(0.0);

            float lightOD       = cloudVolume0LightOD(rPos, 5, cloudLightDir) * sigmaA;
            float skyOD         = cloudVolume0LightOD(rPos, 4, vec3(0.0, 1.0, 0.0)) * sigmaA;

            float powder        = 1.0 - expf(-density * 25.0);

            #ifdef cloudVolumeStoryMode
                float anisoPowder   = 1.0;
            #else
                float anisoPowder   = mix(powder, 1.0, pFade);
            #endif

            vec3 phaseG         = pow(vec3(0.45, 0.25, 0.95), vec3(1.0 + lightOD));

            float phase = cloudPhase(vDotL, 1.0, phaseG);

            stepScatter.x  += max(expf(-lightOD * sigmaT), expf(-lightOD * sigmaT * 0.2) * 0.75) * phase * anisoPowder * sigmaT * 1.25;
            stepScatter.y  += max(expf(-skyOD * sigmaT), expf(-skyOD * sigmaT * 0.2) * 0.75) * powder * sigmaT * 1.25;

            stepScatter     = (sunlight * stepScatter.x * 1.8) + (skylight * stepScatter.y * 1.3);

            float atmosFade = expf(-dist * 2.8e-3);

            stepScatter     = mix(skyColor * sigmaT, stepScatter, atmosFade);

            scattering     += stepScatter * (integral * transmittance);

            transmittance  *= stepT;
        }

        transmittance       = linStep(transmittance, 0.01, 1.0);

        totalScattering    += scattering;
        totalTransmittance *= transmittance;
    }

    #ifdef cloudVolume1Enabled
    visibleVol = worldDir.y > 0.0;

    if (visibleVol) {
        vec3 bottom     = worldDir * ((cloudVolume1Alt - eyeAltitude) * rcp(worldDir.y));
        vec3 top        = worldDir * ((cloudVol1MaxY - eyeAltitude) * rcp(worldDir.y));

        if (worldDir.y < 0.0) {
            bottom      = vec3(0.0);
            top         = vec3(0.0);
        }

        vec3 start      = bottom;
        vec3 end        = top;

        float stepCoeff     = 1.0;
        uint steps          = uint(cloudVolume1Samples * stepCoeff);

        vec3 rStep          = (end - start) * rcp(float(steps));
        vec3 rPos           = rStep * dither + start + camPos;
        float rLength       = length(rStep);

        vec3 scattering     = vec3(0.0);
        float transmittance = 1.0;

        for (uint i = 0; i < steps; ++i, rPos += rStep) {
            if (transmittance < 0.01) break;
            if (rPos.y < cloudVolume1Alt || rPos.y > cloudVol1MaxY) continue;

            float dist  = distance(rPos, camPos);
            if (dist > cloudVolume1Clip) continue;

            float density = cloudVolume1Shape(rPos);
            if (density <= 0.0) continue;

            float extinction    = density * sigmaT;
            float stepT         = exp(-extinction * rLength);
            float integral      = (1.0 - stepT) * rcp(sigmaT);

            vec3 stepScatter    = vec3(0.0);

            float lightOD       = cloudVolume1LightOD(rPos, 4, cloudLightDir) * sigmaA;
            float skyOD         = cloudVolume1LightOD(rPos, 4, vec3(0.0, 1.0, 0.0)) * sigmaA;

            float powder        = 1.0 - expf(-density * 25.0);

            #ifdef cloudVolumeStoryMode
                float anisoPowder   = 1.0;
            #else
                float anisoPowder   = mix(powder, 1.0, pFade);
            #endif

            vec3 phaseG         = pow(vec3(0.45, 0.25, 0.95), vec3(1.0 + lightOD));

            float phase = cloudPhase(vDotL, 1.0, phaseG);

            stepScatter.x  += max(expf(-lightOD * sigmaT), expf(-lightOD * sigmaT * 0.2) * 0.75) * phase * anisoPowder * sigmaT * 1.25;
            stepScatter.y  += max(expf(-skyOD * sigmaT), expf(-skyOD * sigmaT * 0.2) * 0.75) * powder * sigmaT * 1.25;

            stepScatter     = (sunlight * stepScatter.x * 1.6) + (skylight * stepScatter.y * 1.0);

            float atmosFade = expf(-dist * 2.8e-3 * 0.9);

            stepScatter     = mix(skyColor * sigmaT, stepScatter, atmosFade);

            scattering     += stepScatter * (integral * transmittance);

            transmittance  *= stepT;
        }

        transmittance       = linStep(transmittance, 0.01, 1.0);

        totalScattering    += scattering * totalTransmittance;
        totalTransmittance *= transmittance;
    }
    #endif

    return vec4(totalScattering, totalTransmittance);
}

/* ------ ambient occlusion and gi ------ */

#include "/lib/frag/bluenoise.glsl"

#include "/lib/light/ao.glsl"

void main() {
    vec4 sceneColor   = stex(colortex0);
    float sceneDepth = stex(depthtex1).x;

    vec3 viewpos    = screenToViewSpace(vec3(coord, sceneDepth));
    vec3 viewvec    = normalize(viewpos);
    vec3 scenepos   = viewToSceneSpace(viewpos);
    vec3 svec       = normalize(scenepos);

    //no terrain overhead
    //~2fps
    
    if (!landMask(sceneDepth)) {
        vec3 skycol = getSky(viewvec);

        vec3 sun    = getSun(viewvec, sceneColor.rgb);
        vec3 moonstars = getMoon(viewvec, sceneColor.rgb) + skyStars(svec);

        sceneColor.rgb = skycol + (sun+moonstars)*sstep(svec.y, -0.04, 0.01) * (1.0 - wetness * pow(saturate(svec.y), 0.25) * 0.98);
    }

    skyboxData      = vec3(0.0);

    vec2 reflectionCoord    = coord * skyCaptureResolution;

    if (isOnScreenDownscale(reflectionCoord, pixelSize, 4)) {
        vec2 coord          = saturate(reflectionCoord);
        vec3 worldDir       = projectSphere(coord) * vec3(1.0, 1.0, -1.0);
        vec3 viewDir        = mat3(gbufferModelView) * worldDir;

        vec3 skyColor       = getSky(viewDir);

        #ifdef cloudVolumeEnabled
        vec4 clouds     = cloudSystem(worldDir, dot(viewDir, cloudLightDirView), ditherBluenoise(), skyColor);

            skyColor    = skyColor * clouds.a + clouds.rgb;
        #endif

        skyboxData  = skyColor;
    }

    skyboxData  = clamp16F(skyboxData);

    vec4 return3    = vec4(0.0);
    vec3 return6    = vec3(0.0);

    #ifdef lightmapSmoothingEnabled
    if (landMask(sceneDepth)) {
        vec4 tex1   = stex(colortex1);
        return3.xyz = vec3(decode2x8(tex1.z), sceneColor.a);
        return6     = decodeNormal(tex1.xy) * 0.5 + 0.5;
    }
    #endif
    
    #ifdef ambientOcclusionEnabled
        vec2 ao_coord   = (coord)*2.0;

        if (clamp(ao_coord, -0.003, 1.003) == ao_coord) {
            vec2 coord  = ao_coord;
            float d     = depthMax3x3(depthtex1, ao_coord, pixelSize*sqrt(2.0));

            if (landMask(d)) {
                float sceneDepth = stex(depthtex1).x;
                vec4 tex1       = stex(colortex1);

                vec3 sceneNormal = decodeNormal(tex1.xy);
                vec3 viewnormal = normalize(mat3(gbufferModelView) * sceneNormal);

                #ifdef directionalSSAO
                    float ao    = getDSSAO(depthtex1, sceneNormal, sceneDepth, coord, ditherBluenoise());
                #else
                    float ao    = getSSAO(depthtex1, sceneDepth, coord, ditherBluenoise());
                #endif

                return3.w   = ao;
            } else {
                return3.w   = 1.0;
            }
        }
    #endif

    scene   = makeDrawbuffer(sceneColor);
    lightmapData  = clampDrawbuffer(return3);
    filterAux  = clampDrawbuffer(return6);
}